<?php
namespace App\Services;
use Carbon\Carbon;
use App\Models\Admin;
use App\Models\Member;
use App\Models\Campaign;
use App\Models\CampaignDetail;
use DB;

class HrRevenueService{
	public function calculateRevenueForRange(int $hrId, ?string $startDate, ?string $endDate)
	{
		$hr = Admin::findOrFail($hrId);
		$today = now();
		// treat '', 'null', null as "not provided"
		$sd = (is_string($startDate) && trim($startDate) !== '' && strtolower($startDate) !== 'null') ? $startDate : null;
		$ed = (is_string($endDate)   && trim($endDate)   !== '' && strtolower($endDate)   !== 'null') ? $endDate   : null;

		$parseMonth = static function (?string $val, bool $isEnd = false): ?Carbon {
			if (!$val) return null;
			$s = str_replace('/', '-', trim($val));
			try {
				if (preg_match('/^\d{2}-\d{4}$/', $s)){
					$dt = Carbon::createFromFormat('m-Y', $s);
				}
				elseif (preg_match('/^\d{4}-\d{2}$/', $s)){
					$dt = Carbon::createFromFormat('Y-m', $s);
				}
				else{
					$dt = Carbon::parse($s);
				}
			} catch (\Throwable $e) { return null; }
			return $isEnd ? $dt->endOfMonth() : $dt->startOfMonth();
		};
		$start = $sd ? $parseMonth($sd, false) : $today->copy()->startOfMonth();
		$end   = $ed ? $parseMonth($ed, true)  : $today->copy()->endOfMonth();
		if (!$start){
			$start = $today->copy()->startOfMonth();
		}
		if (!$end){
			$end   = $today->copy()->endOfMonth();
		}   
		if ($end->lt($start)){
			[$start, $end] = [$end, $start];
		} 
		
		$total = CampaignDetail::whereHas('campaign', fn($q) => $q->where('state_id', $hr->state_id))
			->whereBetween('call_date', [$start, $end])->where('call_status', 'ANSWERED')->sum(DB::raw('sell_price * CEIL(TIME_TO_SEC(timeSpent) / 60)'));
		return number_format($total, 2);
	}
	public function calculatePersonAchievementForRange(int $hrId, ?string $startDate, ?string $endDate): array
{
    $hr    = \App\Models\Admin::findOrFail($hrId);
    $today = now();

    // ---------- Month tokens (default to current month if not provided) ----------
    $sdToken = (is_string($startDate) && trim($startDate) !== '' && strtolower($startDate) !== 'null')
        ? str_replace('/', '-', trim($startDate))
        : $today->format('m-Y');

    $edToken = (is_string($endDate) && trim($endDate) !== '' && strtolower($endDate) !== 'null')
        ? str_replace('/', '-', trim($endDate))
        : $today->format('m-Y');

    // Parse tokens to month buckets (day=1). No external $parseMonth variable.
    $parseToken = function (string $token): \Carbon\Carbon {
        if (preg_match('/^\d{4}-\d{2}$/', $token)) {
            return \Carbon\Carbon::createFromFormat('Y-m', $token)->day(1);
        }
        if (preg_match('/^\d{2}-\d{4}$/', $token)) {
            return \Carbon\Carbon::createFromFormat('m-Y', $token)->day(1);
        }
        return \Carbon\Carbon::parse($token)->day(1);
    };

    try { $start = $parseToken($sdToken); } catch (\Throwable $e) { $start = $today->copy()->day(1); }
    try { $end   = $parseToken($edToken); } catch (\Throwable $e) { $end   = $today->copy()->day(1); }

    if ($end->lt($start)) { [$start, $end] = [$end, $start]; }

    // ---------- Build YYYY-MM list (inclusive) ----------
    $months = [];
    for ($cur = $start->copy(); $cur->year < $end->year || ($cur->year === $end->year && $cur->month <= $end->month); $cur->addMonth()) {
        $months[] = $cur->format('Y-m');
    }
    if (empty($months)) { $months[] = $start->format('Y-m'); }

    // ---------- REQUIRED per role from targets across the months ----------
    $targets = $hr->targets()
        ->where(function ($q) use ($months) {
            foreach ($months as $ym) {
                [$y, $m] = explode('-', $ym);
                $q->orWhere(function ($qq) use ($y, $m) {
                    $qq->where('year', $y)->whereRaw("LPAD(month, 2, '0') = ?", [$m]);
                });
            }
        })->get();

    $requiredByRoleId = []; // [role_id => required_count]
    foreach ($targets as $t) {
        $idsCsv = (string) $t->target_for_role; // "21,22,23"
        $reqCsv = (string) $t->required_role;   // "50,1,1"
        if ($idsCsv === '' || $reqCsv === '') continue;

        $ids = array_map('trim', explode(',', $idsCsv));
        $req = array_map('trim', explode(',', $reqCsv));
        foreach ($ids as $i => $rid) {
            if (!isset($req[$i]) || $req[$i] === '' || !is_numeric($req[$i])) continue;
            $count = (int) $req[$i];
            if ($count <= 0) continue;
            $rid = (int) $rid;
            $requiredByRoleId[$rid] = ($requiredByRoleId[$rid] ?? 0) + $count;
        }
    }

    if (empty($requiredByRoleId)) {
        return [
            'required_total' => 0,
            'achieved_total' => 0,
            'met_all'        => false,
            'by_role'        => [],
            'text'           => 'Total 0',
            'html'           => 'Total <b>0</b>',
        ];
    }

    // ---------- ACHIEVED per role (new employees added in range) ----------
    $rangeStart = \Carbon\Carbon::createFromFormat('Y-m', $months[0])->startOfMonth();
    $rangeEnd   = \Carbon\Carbon::createFromFormat('Y-m', end($months))->endOfMonth();

    $employeesQ       = \App\Models\Admin::query();
    $employeesTblName = $employeesQ->getModel()->getTable();

    $achievedQuery = (clone $employeesQ)->whereBetween('created_at', [$rangeStart, $rangeEnd])->whereIn('role_id', array_keys($requiredByRoleId));

    // Optional filters if columns exist
    if (\Illuminate\Support\Facades\Schema::hasColumn($employeesTblName, 'author_id')) {
        $achievedQuery->where('author_id', $hrId);
    }
    if (\Illuminate\Support\Facades\Schema::hasColumn($employeesTblName, 'state_id') && !is_null($hr->state_id)) {
        $achievedQuery->where('state_id', $hr->state_id);
    }
    if (\Illuminate\Support\Facades\Schema::hasColumn($employeesTblName, 'deleted_at')) {
        $achievedQuery->whereNull('deleted_at');
    }

    $achievedByRoleId = $achievedQuery->selectRaw('role_id as rid, COUNT(*) as c')->groupBy('rid')->pluck('c', 'rid')->all();

    // ---------- Map role IDs → names (no hardcoded column list) ----------
    $roles = \App\Models\Role::whereIn('id', array_keys($requiredByRoleId))->get();
    $nameById = [];
    foreach ($roles as $r) {
        $nameById[$r->id] = $r->name
            ?? $r->role_name
            ?? $r->title
            ?? $r->label
            ?? $r->slug
            ?? ('Role#'.$r->id);
    }

    // ---------- Build rows + totals ----------
    $rows     = [];
    $reqTotal = 0;
    $achTotal = 0;

    foreach ($requiredByRoleId as $rid => $reqCount) {
        $achCount = (int) ($achievedByRoleId[$rid] ?? 0);
        $reqTotal += (int) $reqCount;
        $achTotal += $achCount;

        $rows[] = [
            'role_id'   => $rid,
            'role'      => $nameById[$rid] ?? ('Role#'.$rid),
            'required'  => (int) $reqCount,
            'achieved'  => $achCount,
            'remaining' => max(0, (int)$reqCount - $achCount),
            'met'       => $achCount >= (int)$reqCount,
        ];
    }

    // Unmet first, then alphabetically
    usort($rows, function ($a, $b) {
        if ($a['met'] !== $b['met']) return $a['met'] ? 1 : -1;
        return strnatcasecmp($a['role'], $b['role']);
    });

    $metAll = ($achTotal >= $reqTotal) && collect($rows)->every(fn($r) => $r['met']);

    // ---------- Human friendly strings ----------
    $textParts = [];
    foreach ($rows as $r) {
        $textParts[] = "{$r['role']} {$r['achieved']}/{$r['required']}";
    }
    $text = (empty($textParts) ? '' : implode(', ', $textParts) . ' → ') . "Total {$achTotal}/{$reqTotal}" . ($metAll ? ' ✓' : ' ✗');

    $htmlLines = [];
    foreach ($rows as $r) {
        $mark  = $r['met'] ? '✓' : '✗';
        $color = $r['met'] ? '#16a34a' : '#dc2626'; // green/red
        $bg    = $r['met'] ? '#ecfdf5' : '#fef2f2'; // light green/red background

        $htmlLines[] = '
            <div style="
                background: '.$bg.';
                border: 1px solid '.$color.';
                border-radius: 12px;
                padding: 4px 16px;
                margin: 8px 0 0;
                font-family: Arial, sans-serif;
                box-shadow: 0 2px 6px rgba(0,0,0,0.08);
                display: flex;
                justify-content: space-between;
                align-items: center;
            ">
                <div style="font-size: 11px; color:#000;">
                    <b>'.e($r['role']).'</b> <br>
                    <span style="color:#000; font-weight: 700;">'.$r['achieved'].' / '.$r['required'].'</span>
                </div>
                <div style="font-size: 14px; font-weight:bold; color:'.$color.';">
                    '.$mark.'
                </div>
            </div>
        ';
    }
    $html = implode('', $htmlLines)
          . '<br> <b> Total : '.$achTotal.'</b>/<b>'.$reqTotal.'</b> '
          . '<span style="color:'.($metAll ? '#16a34a' : '#dc2626').'">'.($metAll ? '✓' : '✗').'</span>';

    return [
        'required_total' => $reqTotal,
        'achieved_total' => $achTotal,
        'met_all'        => $metAll,
        'by_role'        => $rows,
        'text'           => $text,
        'html'           => $html,
    ];
}
    /*public function calculateRevenueForRange(int $hrId, ?string $startDate, ?string $endDate){
        $hr = Admin::findOrFail($hrId);
        $today = now();
        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $today->copy()->startOfMonth();
        $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $today->copy()->endOfDay();
        $total=CampaignDetail::whereHas('campaign', function ($q) use ($hr) {
                    $q->where('state_id', $hr->state_id);
                })
				->whereBetween('call_date', [$start, $end])->where('call_status','ANSWERED')
				->sum(DB::raw('sell_price * CEIL(TIME_TO_SEC(timeSpent) / 60)'));
		return number_format($total, 2);
	}*/
	public function RevenueForTelecallerManager(int $hrId, ?string $startDate, ?string $endDate){
        $hr = Admin::findOrFail($hrId);
        $today = now();
        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $today->copy()->startOfMonth();
        $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $today->copy()->endOfDay();
        /*return CampaignDetail::whereHas('campaign', function ($q) use ($hr) {
                    $q->where('state_id', $hr->state_id);
                })*/
		$adminCities = explode(',', $hr->jobcity);
		$query = CampaignDetail::whereHas('campaign', function ($q) use ($adminCities) {
			$q->where(function ($query) use ($adminCities) {
				foreach ($adminCities as $city) {
					$query->orWhereRaw('FIND_IN_SET(?, city)', [$city]);
				}
			});
		})->whereBetween('call_date', [$start, $end]);
		/*dd(vsprintf(
			str_replace('?', "'%s'", $query->toSql()),
			$query->getBindings()
		));*/
		$total = $query->sum('sell_price');

		return number_format($total, 2);
	}
	public function calculateTelecallerRevenueForRange(int $emp_id, ?string $startDate, ?string $endDate){
        $hr = Admin::findOrFail($emp_id);
        $today = now();
        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $today->copy()->startOfMonth();
        $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $today->copy()->endOfDay();
        /*$total=CampaignDetail::whereHas('campaign', function ($q) use ($hr) {
                    $q->where('state_id', $hr->state_id);
                })
				->whereBetween('call_date', [$start, $end])
				->sum('sell_price');*/
		$total = CampaignDetail::where('staff_id', $emp_id)->whereBetween('call_date', [$start, $end])->sum('sell_price');
		return number_format($total, 2);
	}
	public function RevenueForCompanyMarketingAgencyManger(int $hrId, ?string $startDate, ?string $endDate){
        $today = now();
        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $today->copy()->startOfMonth();
        $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $today->copy()->endOfDay();
		$memberIds = Member::where('admin_referred_id', $hrId)->pluck('id');
		
        $query = CampaignDetail::whereIn('member_id', $memberIds)->whereBetween('call_date', [$start, $end]);
		/*dd(vsprintf(
			str_replace('?', "'%s'", $query->toSql()),
			$query->getBindings()
		));*/
		$total = $query->sum('sell_price');
		return number_format($total, 2);
	}
	public function RevenueForCompanyMarketingEmployee(int $emp_id, ?string $startDate, ?string $endDate){
        $today = now();
        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $today->copy()->startOfMonth();
        $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $today->copy()->endOfDay();
		$memberIds = Member::where('admin_referred_id', $emp_id)->pluck('id');
		
        $query = Campaign::whereIn('member_id', $memberIds)->whereBetween('created_at', [$start, $end])->where('status', '!=', 'Reject');
		$total = $query->sum('approx_cost');
		/*dd(vsprintf(
			str_replace('?', "'%s'", $query->toSql()),
			$query->getBindings()
		));
		$total = $query->sum('approx_cost');*/
		return number_format($total, 2);
	}
}

 