<?php
 
namespace App\Services;
use Carbon\Carbon;
use App\Models\Admin;
use App\Models\Role;

class EmployeeTargetsForRange{

// use App\Models\Role; // adjust import if needed

public function calculateTargetsForRange(int $hrId, ?string $startDate = null, ?string $endDate = null)
{
    $hr = Admin::with(['targets'])->findOrFail($hrId);
    $today = now();

    // Parse "mm-yyyy" (primary). Also tolerates "yyyy-mm" / "August 2025".
    $parseMonthToFirst = static function (?string $val): ?Carbon {
        if (!$val) return null;
        $s = str_replace('/', '-', trim($val));
        try {
            if (preg_match('/^\d{2}-\d{4}$/', $s)) {            // mm-yyyy
                return Carbon::createFromFormat('m-Y', $s)->day(1);
            } elseif (preg_match('/^\d{4}-\d{2}$/', $s)) {      // yyyy-mm
                return Carbon::createFromFormat('Y-m', $s)->day(1);
            }
            return Carbon::parse($s)->day(1);                   // fallback
        } catch (\Throwable $e) {
            return null;
        }
    };

    // "Admin=1, HR=2" formatter
	$makeBreakdownHtml = static function (array $roleCounts, int $total): string {
		if (empty($roleCounts)) {
			return '<div class="t-total fw-bold">Total : <span class="text-black">' . $total . '</span></div>';
		}
		ksort($roleCounts, SORT_NATURAL | SORT_FLAG_CASE);

		$badges = '';
		foreach ($roleCounts as $role => $cnt) {
			// truncate very long names to keep pill neat (optional)
			$label = mb_strimwidth($role, 0, 28, '…');
			$badges .= '<span class="badge-role" title="' . e($role) . '">'
					.  e($label) . '<span class="count">' . (int)$cnt . '</span></span>';
		}
		return '<div class="role-badges">'.$badges.'</div>'
			 . '<div class="t-total fw-bold">Total : <span class="text-black">' . $total . '</span></div>';
	};
    $makeBreakdown = static function (array $roleCounts): string {
        if (empty($roleCounts)) return '';
        ksort($roleCounts, SORT_NATURAL | SORT_FLAG_CASE);
        $parts = [];
        foreach ($roleCounts as $role => $cnt) {
            $parts[] = "{$role}={$cnt}";
        }
        return implode(', ', $parts);
    };

    // === Case 1: no dates → current month only ===
    if (is_null($startDate) && is_null($endDate)) {
        $currentMonth = (int) $today->format('m');
        $year = (int) $today->format('Y');

        $target = $hr->targets()
            ->where('month', $currentMonth)
            ->where('year',  $year)
            ->first();

        if (!$target) {
            return ['revenue' => 0, 'person' => 'total 0'];
        }

        // Per-role breakdown for this month
        $idsCsv = (string) $target->target_for_role;   // role IDs CSV
        $reqCsv = (string) $target->required_role;     // counts CSV (same positions)
        $roleCountsAgg = [];
        $allRoleIds = [];

        if ($idsCsv !== '' && $reqCsv !== '') {
            $ids = array_map('trim', explode(',', $idsCsv));
            $req = array_map('trim', explode(',', $reqCsv));
            foreach ($ids as $i => $rid) {
                if (!isset($req[$i]) || $req[$i] === '' || !is_numeric($req[$i])) continue;
                $cnt = (int) $req[$i];
                if ($cnt <= 0) continue;
                $rid = (int) $rid;
                $allRoleIds[] = $rid;
                $roleCountsAgg[$rid] = ($roleCountsAgg[$rid] ?? 0) + $cnt;
            }
        }

        // Map role IDs → names (don't select non-existent columns)
        $nameCounts = [];
        if (!empty($roleCountsAgg)) {
            $uniqueIds = array_values(array_unique($allRoleIds));
            $roles = \App\Models\Role::whereIn('id', $uniqueIds)->get(); // <-- no ['id','name','role']
            $nameById = [];
            foreach ($roles as $r) {
                // try common name columns; nulls are fine
                $nameById[$r->id] = $r->name
                    ?? $r->role_name
                    ?? $r->title
                    ?? $r->label
                    ?? $r->slug
                    ?? ('Role#'.$r->id);
            }
            foreach ($roleCountsAgg as $rid => $cnt) {
                $name = $nameById[$rid] ?? ('Role#'.$rid);
                $nameCounts[$name] = ($nameCounts[$name] ?? 0) + $cnt;
            }
        }

        $breakdownCore = $makeBreakdown($nameCounts);
        $personTotal   = (int) round((float) $target->target_by_person); // stored total
        $personText    = $breakdownCore !== '' ? "{$breakdownCore} → total {$personTotal}" : "total {$personTotal}";
		$personHtml = $makeBreakdownHtml($nameCounts, $personTotal);

        return [
            'revenue' => round((float) $target->target_by_revenue, 2),
            'person'  => $personText,
			'person_html'  => $personHtml, 
        ];
    }

    // === Case 2: dates provided → month buckets only (no day math) ===
    $start = $startDate ? $parseMonthToFirst($startDate) : $today->copy()->day(1);
    $end   = $endDate   ? $parseMonthToFirst($endDate)   : $today->copy()->day(1);

    if (!$start) $start = $today->copy()->day(1);
    if (!$end)   $end   = $today->copy()->day(1);

    if ($end->lt($start)) {
        [$start, $end] = [$end, $start];
    }

    // Build list of YYYY-MM months inclusive
    $months = [];
    $cursor = $start->copy();
    while ($cursor->year < $end->year || ($cursor->year === $end->year && $cursor->month <= $end->month)) {
        $months[] = $cursor->format('Y-m');
        $cursor->addMonth();
    }

    // Fetch targets for those months
    $targets = $hr->targets()
        ->where(function ($query) use ($months) {
            foreach ($months as $ym) {
                [$y, $m] = explode('-', $ym);
                $query->orWhere(function ($q) use ($y, $m) {
                    $q->where('year', $y)
                      ->whereRaw("LPAD(month, 2, '0') = ?", [$m]);
                });
            }
        })
        ->get()
        ->keyBy(function ($item) {
            return sprintf('%d-%02d', $item->year, $item->month);
        });

    $totalRevenueTarget = 0.0;
    $totalPersonTarget  = 0.0;

    // Aggregate role counts across the range
    $roleCountsAgg = [];
    $allRoleIds = [];

    foreach ($months as $ym) {
        $monthly = $targets->get($ym);
        if (!$monthly) continue;

        $totalRevenueTarget += (float) $monthly->target_by_revenue;
        $totalPersonTarget  += (float) $monthly->target_by_person; // stored monthly totals

        $idsCsv = (string) $monthly->target_for_role;
        $reqCsv = (string) $monthly->required_role;
        if ($idsCsv === '' || $reqCsv === '') continue;

        $ids = array_map('trim', explode(',', $idsCsv));
        $req = array_map('trim', explode(',', $reqCsv));
        foreach ($ids as $i => $rid) {
            if (!isset($req[$i]) || $req[$i] === '' || !is_numeric($req[$i])) continue;
            $cnt = (int) $req[$i];
            if ($cnt <= 0) continue;
            $rid = (int) $rid;
            $allRoleIds[] = $rid;
            $roleCountsAgg[$rid] = ($roleCountsAgg[$rid] ?? 0) + $cnt;
        }
    }

    // Map role IDs → names (safe, no explicit column list)
    $nameCounts = [];
    if (!empty($roleCountsAgg)) {
        $uniqueIds = array_values(array_unique($allRoleIds));
        $roles = \App\Models\Role::whereIn('id', $uniqueIds)->get(); // <-- removed ['id','name','role']
        $nameById = [];
        foreach ($roles as $r) {
            $nameById[$r->id] = $r->name
                ?? $r->role_name
                ?? $r->title
                ?? $r->label
                ?? $r->slug
                ?? ('Role#'.$r->id);
        }
        foreach ($roleCountsAgg as $rid => $cnt) {
            $name = $nameById[$rid] ?? ('Role#'.$rid);
            $nameCounts[$name] = ($nameCounts[$name] ?? 0) + $cnt;
        }
    }

    $breakdownCore = $makeBreakdown($nameCounts);
    $personTotal   = (int) round($totalPersonTarget);
    $personText    = $breakdownCore !== '' ? "{$breakdownCore} → total {$personTotal}" : "total {$personTotal}";
	$personHtml = $makeBreakdownHtml($nameCounts, $personTotal);

    return [
        'revenue' => round($totalRevenueTarget, 2),
        'person'  => $personText,
        'person_html'  => $personHtml,  
    ];
}

    /*public function calculateTargetsForRange(int $hrId, ?string $startDate = null, ?string $endDate = null)
    {
        $hr = Admin::with(['targets'])->findOrFail($hrId);
        $today = now();
        
        // If both dates are null, return current month's target if it exists
        if (is_null($startDate) && is_null($endDate)) {
            $currentMonth = $today->format('m');
            $year = $today->format('Y');
            $target = $hr->targets()->where('month', $currentMonth)->where('year', $year)->first();
            if ($target) {
                return [
					'revenue' => round($target->target_by_revenue, 2),
                    'person' => round($target->target_by_person, 2),
                ];
            }
            return ['revenue' => 0, 'person' => 0];
        }
        // Use provided date range or default to current month
        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $today->copy()->startOfMonth();
        $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $today->copy()->endOfMonth();
        
        $months = [];
        $current = $start->copy()->startOfMonth();
        while ($current <= $end) {
            $months[] = $current->format('Y-m');
            $current->addMonth();
        }
        
        // Fetch targets for the months in the range
        $targets = $hr->targets()
            ->where(function ($query) use ($months) {
                foreach ($months as $month) {
                    $yearMonth = explode('-', $month);
                    $query->orWhere(function ($q) use ($yearMonth) {
                        $q->where('year', $yearMonth[0])
                        ->whereRaw("LPAD(month, 2, '0') = ?", [$yearMonth[1]]);
                    });
                }
            })
            ->get()
            ->keyBy(function ($item) {
                return sprintf('%d-%02d', $item->year, $item->month);
            });
        
        $totalRevenueTarget = 0;
        $totalPersonTarget = 0;
        
        foreach ($months as $month) {
            [$year, $mon] = explode('-', $month);
            $monthStart = Carbon::createFromDate($year, $mon, 1);
            $monthEnd = $monthStart->copy()->endOfMonth();
            
            $periodStart = $start->greaterThan($monthStart) ? $start : $monthStart;
            $periodEnd = $end->lessThan($monthEnd) ? $end : $monthEnd;
            
            $daysInPeriod = $periodStart->diffInDays($periodEnd) + 1;
            $daysInMonth = $monthStart->daysInMonth;
            
            $key = sprintf('%d-%02d', $year, $mon);
            $monthlyTarget = $targets->get($key);
            if ($monthlyTarget) {
                $fullMonthTarget = ($periodStart->eq($monthStart) && $periodEnd->eq($monthEnd));
                $RevenuetargetValue = $monthlyTarget->target_by_revenue;
                $PersontargetValue = $monthlyTarget->target_by_person;
				
                $totalRevenueTarget += $fullMonthTarget ? $RevenuetargetValue : ($RevenuetargetValue / $daysInMonth) * $daysInPeriod;
				$totalPersonTarget += $fullMonthTarget ? $PersontargetValue : ($PersontargetValue / $daysInMonth) * $daysInPeriod;
            }
        }
        
        return [
            'revenue' => round($totalRevenueTarget, 2),
            'person' => round($totalPersonTarget, 2)
        ];
    }*/

}

 