<?php
 
namespace App\Services;
 
use Carbon\Carbon;

use App\Models\Member;
 
class AgencyTargetsForRange{
    
    public function calculateTargetsForRange(int $hrId, ?string $startDate = null, ?string $endDate = null){
        $hr = Member::with(['targets'])->findOrFail($hrId);
        $today = now();
        
        // If both dates are null, return current month's target if it exists
        if (is_null($startDate) && is_null($endDate)) {
            $currentMonth = $today->format('m');
            $year = $today->format('Y');
            $target = $hr->targets()
                ->where('month', $currentMonth)
                ->where('year', $year)
                ->first();
                
            if ($target) {
                return [
                    'revenue' => $target->by_target == 'by_revenue' ? round($target->target, 2) : 0,
                    'person' => $target->by_target == 'by_person' ? round($target->target, 2) : 0
                ];
            }
            return ['revenue' => 0, 'person' => 0];
        }
        
        // Use provided date range or default to current month
        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $today->copy()->startOfMonth();
        $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $today->copy()->endOfMonth();
        
        $months = [];
        $current = $start->copy()->startOfMonth();
        
        while ($current <= $end) {
            $months[] = $current->format('Y-m');
            $current->addMonth();
        }
        
        // Fetch targets for the months in the range
        $targets = $hr->targets()
            ->where(function ($query) use ($months) {
                foreach ($months as $month) {
                    $yearMonth = explode('-', $month);
                    $query->orWhere(function ($q) use ($yearMonth) {
                        $q->where('year', $yearMonth[0])
                        ->whereRaw("LPAD(month, 2, '0') = ?", [$yearMonth[1]]);
                    });
                }
            })
            ->get()
            ->keyBy(function ($item) {
                return sprintf('%d-%02d', $item->year, $item->month);
            });
        
        $totalRevenueTarget = 0;
        $totalPersonTarget = 0;
        
        foreach ($months as $month) {
            [$year, $mon] = explode('-', $month);
            $monthStart = Carbon::createFromDate($year, $mon, 1);
            $monthEnd = $monthStart->copy()->endOfMonth();
            
            $periodStart = $start->greaterThan($monthStart) ? $start : $monthStart;
            $periodEnd = $end->lessThan($monthEnd) ? $end : $monthEnd;
            
            $daysInPeriod = $periodStart->diffInDays($periodEnd) + 1;
            $daysInMonth = $monthStart->daysInMonth;
            
            $key = sprintf('%d-%02d', $year, $mon);
            $monthlyTarget = $targets->get($key);
            
            if ($monthlyTarget) {
                $fullMonthTarget = ($periodStart->eq($monthStart) && $periodEnd->eq($monthEnd));
                $targetValue = $monthlyTarget->target;
                
                if ($monthlyTarget->by_target == 'by_revenue') {
                    $totalRevenueTarget += $fullMonthTarget ? $targetValue : ($targetValue / $daysInMonth) * $daysInPeriod;
                }
                
                if ($monthlyTarget->by_target == 'by_person') {
                    $totalPersonTarget += $fullMonthTarget ? $targetValue : ($targetValue / $daysInMonth) * $daysInPeriod;
                }
            }
        }
        
        return [
            'revenue' => round($totalRevenueTarget, 2),
            'person' => round($totalPersonTarget, 2)
        ];
    }

}

 