<?php

namespace App\Http\Controllers\AdminAuth;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\City;
use App\Models\Role;
use App\Models\State;
use Auth;
use DB;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;


class UserController extends Controller
{
    private $data = [
        'route' => 'admin.user.',
        'title' => 'Office Employee',
        'menu' => 'Office Employee',
        'submenu' => '',
    ];

    public function index(Request $request)
    {
        if (! rolemanagement('User', 1)) {
            abort(401);
        }
        if ($request->ajax()) {
            $records = Admin::orderBy('id', 'DESC')->with(['role', 'citys']);
            $records = restrictByAuthor($records);

            return datatables()->of($records)
                ->addIndexColumn()
                ->editColumn('role', function ($record) {
                    $role = Role::where('id', $record->role_id)->first();

                    return $role->name ?? '-';
                })
                ->editColumn('city', function ($record) {
                    return $record->citys->name ?? '-';
                })
                ->editColumn('state_id', function ($record) {
                    return $record->state->name ?? '-';
                })
                ->editColumn('created_at', function ($record) {
                    return date('d-m-Y', strtotime($record->created_at));
                })
                ->editColumn('name', function ($record) {
                    $now = now();

                    $html = '<div class="user-pro-body custom_width_user_profile" style="margin-right: 16px;">';
                    $html .= '<img alt="user-img" class="brround" src="'.url($record->profile_image ?? 'assets/admin/images/user_av.png').'">';

                    $html .= '</div>';
                    if (rolemanagement('User', 3)) {
                        $html .= '<a href="'.url('secure-dashboard/user/'.$record->id.'/edit').'" data-bs-toggle="tooltip" title="" data-bs-original-title="edit">'.htmlspecialchars($record->name ?? '-').'</a> ';
                    } else {
                        $html .= htmlspecialchars($record->name ?? '-');
                    }

                    return $html;
                })
                ->editColumn('status', function ($record) {
                    return '<div class="form-check form-switch ">
                            <input class="form-check-input status-toggle" style="" id="statusToggle" type="checkbox" data-id="'.$record->id.'" data-status="'.$record->status.'" '.($record->status == 'Active' ? 'checked' : '').'></div>';
                })
                ->addColumn('action', function ($record) {
                    $html = '';
                    $html = '<div style="display: flex;  gap: 5px; align-items: center; justify-content: center;">';
                    if (rolemanagement('User', 3)) {
                        $html .= '<a href="'.url('secure-dashboard/user/'.$record->id.'/edit').'" class="btn btn-sm btn-info btn-b" data-bs-toggle="tooltip" title="" data-bs-original-title="edit"><i class="fas fa-edit"></i></a> ';
                    }
                    if (rolemanagement('User', 4)) {
                        $html .= '<button  class="btn btn-sm btn-danger data-delete" data-id="'.$record->id.'"  data-bs-toggle="tooltip" title="" data-bs-original-title="delete"><i class="fas fa-trash"></i></button>';
                    }
                    $html .= '</div>';

                    return $html;
                })
                ->rawColumns(['status', 'sequence', 'action', 'name'])
                ->make(true);
        }

        return view('admin.user.index', $this->data);

    }

    public function create()
    {
        if (! rolemanagement('User', 2)) {
            abort(401);
        }

        $this->data['title'] = 'Create User';
        $this->data['user'] = [];
        $this->data['citys'] = City::where('status', 'Active')->orderBy('name', 'asc')->get();
        $this->data['roles'] = Role::where('status', 'Active')->get();
        $this->data['states'] = State::where('status', 'Active')->OrderBySequence('asc')->get();

        return view('admin.user.create', $this->data);
    }

    public function store(Request $request)
    {
        $rules = [
            'role' => 'required|exists:roles,id',
            'state_id' => 'required|exists:states,id',
            'name' => 'required|string|max:255',
            'city' => 'required|exists:cities,id',
            'pancard' => 'required|string|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/',
            'aadharcard' => 'required|numeric|digits:12',
            'qualification' => 'required|string|max:255',
            'address' => 'required|string|max:1000',
            'ifsc_code' => 'required|string|regex:/^[A-Z]{4}0[A-Z0-9]{6}$/',
            'bank_name' => 'required|string|max:255',
            'account_number' => 'required|numeric',
            'mobile' => 'required|numeric|digits:10|unique:admins,mobile',
            'email' => 'required|email|unique:admins,email|max:255',
            'password' => 'required|min:6',

            'aadhar_img' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
            'pancard_img' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
            'selfi_image' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
            'passbook_image' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();
        try {
            $data = [
                'role_id' => $request->role,
                'name' => $request->name,
                'city' => $request->city,
                'state_id' => $request->state_id,
                'pancard' => $request->pancard,
                'aadharcard' => $request->aadharcard,
                'qualification' => $request->qualification,
                'address' => $request->address,
                'ifsc_code' => $request->ifsc_code,
                'bank_name' => $request->bank_name,
                'account_number' => $request->account_number,
                'mobile' => $request->mobile,
                'email' => $request->email,
                'password' => bcrypt($request->password),
                'show_password' => $request->password,
                'author_id' => auth('admin')->user()->role_id != 1 ? auth('admin')->id() : null,
            ];

            $files = ['aadhar_img', 'pancard_img', 'selfi_image', 'passbook_image'];

            foreach ($files as $f) {
                if ($request->hasFile($f)) {
                    $fileName = $f.'_'.rand().time().'.'.$request->file($f)->extension();
                    $request->file($f)->move(public_path('uploads/users/documents'), $fileName);
                    $data[$f] = 'uploads/users/documents/'.$fileName;
                }
            }

            $user = Admin::create($data);

            $user->employee_no = 'TURBO'.date('Y').$user->id;
            $user->save();

            DB::commit();

            return redirect('secure-dashboard/user')->withSuccess('User added successfully');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->withError('Error: '.$e->getMessage())->withInput();
        }
    }

    public function edit($id)
    {
        $this->data['title'] = 'Edit User';
        $this->data['user'] = restrictByAuthor(Admin::where('id', $id))->firstOrFail();
        $this->data['citys'] = City::where('status', 'Active')->get();
        $this->data['roles'] = Role::where('status', 'Active')->get();
        $this->data['states'] = State::where('status', 'Active')->OrderBySequence('asc')->get();

        return view('admin.user.create', $this->data);
    }

    public function update(Request $request, $id)
    {
        $user = restrictByAuthor(Admin::where('id', $id))->firstOrFail();

        $rules = [
            'role' => 'required|exists:roles,id',
            'state_id' => 'required|exists:states,id',
            'name' => 'required|string|max:255',
            'city' => 'required|exists:cities,id',
            'pancard' => 'required|string|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/',
            'aadharcard' => 'required|numeric|digits:12',
            'qualification' => 'required|string|max:255',
            'address' => 'required|string|max:1000',
            'ifsc_code' => 'required|string|regex:/^[A-Z]{4}0[A-Z0-9]{6}$/',
            'bank_name' => 'required|string|max:255',
            'account_number' => 'required|numeric',
            'mobile' => ['required', 'numeric', 'digits:10', Rule::unique('admins', 'mobile')->ignore($id)],
            'email' => ['required', 'email', Rule::unique('admins', 'email')->ignore($id)],
            'password' => 'nullable|min:6',

            'aadhar_img' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
            'pancard_img' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
            'selfi_image' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
            'passbook_image' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx|max:5120',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();
        try {
            $data = [
                'role_id' => $request->role,
                'name' => $request->name,
                'city' => $request->city,
                'state_id' => $request->state_id,
                'pancard' => $request->pancard,
                'aadharcard' => $request->aadharcard,
                'qualification' => $request->qualification,
                'address' => $request->address,
                'ifsc_code' => $request->ifsc_code,
                'bank_name' => $request->bank_name,
                'account_number' => $request->account_number,
                'mobile' => $request->mobile,
                'email' => $request->email,
                'password' => $request->filled('password') ? bcrypt($request->password) : $user->password,
                'show_password' => $request->filled('password') ? $request->password : $user->show_password,
            ];

            $files = ['aadhar_img', 'pancard_img', 'selfi_image', 'passbook_image'];

            // foreach ($files as $f) {
            //     if ($request->hasFile($f)) {
            //         if (!empty($record[$f]) && file_exists(public_path($record[$f]))) {
            //             unlink(public_path($record[$f]));
            //         }
            //         $fileName = $f . '_' . rand() . time() . '.' . $request->file($f)->extension();
            //         $request->file($f)->move(public_path('uploads/users/documents'), $fileName);
            //         $data[$f] = 'uploads/users/documents/' . $fileName;
            //     }
            // }

            $user->fill($data);
            if ($user->isDirty()) {
                $user->save();
            }

            DB::commit();

            return redirect('secure-dashboard/user')->withSuccess('User updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->withError('Error: '.$e->getMessage())->withInput();
        }
    }

    public function destroy(Request $request)
    {
        if (! rolemanagement('User', 4)) {
            abort(401);
        }

        $user = restrictByAuthor(Admin::where('id', $request->id))->first();

        if ($user) {
            DB::beginTransaction();
            try {
                $userName = $user->name;
                $userId = $user->id;
                $user->delete();
                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'User has been deleted successfully.',
                ]);
            } catch (\Exception $e) {
                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred while deleting the user: '.$e->getMessage(),
                ]);
            }
        }

        return response()->json([
            'success' => false,
            'message' => 'User not found',
        ]);
    }

    public function checkUserMobile(Request $request)
    {
        $request->validate([
            'transfer_type' => 'required|in:employee,agency',
            'mobile' => 'required',
        ]);

        if ($request->transfer_type === 'employee') {
            $admin = Admin::where('mobile', $request->mobile)->first();

            if (! $admin) {
                return response()->json(['success' => false, 'message' => 'No employee found with this mobile number.']);
            }

            return response()->json([
                'success' => true,
                'name' => $admin->name ?? 'Unknown Employee',
            ]);

        } else {
            $member = Member::where('mobile', $request->mobile)->first();

            if (! $member) {
                return response()->json(['success' => false, 'message' => 'No agency found with this mobile number.']);
            }

            return response()->json([
                'success' => true,
                'name' => $member->name ?? 'Unknown Agency',
            ]);
        }
    }

    public function transferEmployee(Request $request)
    {
        $request->validate([
            'transfer_type' => 'required|in:employee,agency',
            'mobile' => 'required',
            'password' => 'required',
        ]);

        $loginId = Auth::guard('admin')->id();
        $loginName = Auth::guard('admin')->user()->name;

        if ($request->transfer_type === 'employee') {
            $admin = Admin::where('mobile', $request->mobile)->first();

            if (! $admin) {
                return response()->json(['success' => false, 'message' => 'Mobile number not found in Admin.']);
            }

            if (! Hash::check($request->password, $admin->password)) {
                return response()->json(['success' => false, 'message' => 'Invalid password.']);
            }

            if ($admin->id == $loginId) {
                return response()->json(['success' => false, 'message' => 'You cannot transfer yourself under yourself!']);
            }

            if ($admin->id == Auth::guard('admin')->user()->author_id) {
                return response()->json(['success' => false, 'message' => 'You cannot transfer your parent under you!']);
            }

            if ($admin->author_id) {
                if ($admin->author_id == $loginId) {
                    return response()->json(['success' => false, 'message' => 'Already transferred under you!']);
                }

                return response()->json(['success' => false, 'message' => 'This Employee is already assigned to another Employee!']);
            }

            $oldAuthor = $admin->author_id;
            $admin->author_id = $loginId;

            if ($admin->isDirty('author_id')) {
                $admin->save();
            }

            return response()->json([
                'success' => true,
                'message' => 'Employee transfer successful!',
            ]);
        } else {
            $member = Member::where('mobile', $request->mobile)->first();

            if (! $member) {
                return response()->json(['success' => false, 'message' => 'Mobile number not found in Member.']);
            }

            if (! Hash::check($request->password, $member->password)) {
                return response()->json(['success' => false, 'message' => 'Invalid password.']);
            }

            if ($member->member_referred_id) {
                return response()->json(['success' => false, 'message' => 'This Member is already assigned to another Member!']);
            }

            if ($member->admin_referred_id) {
                if ($member->admin_referred_id == $loginId) {
                    return response()->json(['success' => false, 'message' => 'Already transferred under you!']);
                }

                return response()->json(['success' => false, 'message' => 'This Member is already assigned to another Employee!']);
            }

            $oldRef = $member->admin_referred_id;
            $member->admin_referred_id = $loginId;
            $member->member_referred_id = null;

            if ($member->isDirty(['admin_referred_id', 'member_referred_id'])) {
                $member->save();
            }

            return response()->json([
                'success' => true,
                'message' => 'Agency transfer successful!',
            ]);
        }
    }
}
