<?php

namespace App\Http\Controllers\AdminAuth;

use App\Http\Controllers\Controller;
use App\Models\RawMaterial;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class RawMaterialController extends Controller
{
    private $data = [
        'route' => 'admin.raw_materials.',
        'title' => 'Raw Materials',
        'menu' => 'Raw Materials',
        'submenu' => '',
    ];

    public function index(Request $request)
    {
        if (! rolemanagement('Raw Material', 1)) {
            abort(401);
        }

        if ($request->ajax()) {

            $records = RawMaterial::with(['category', 'purchaseGroup'])
                ->orderBy('id', 'DESC')
                ->select('raw_materials.*');

            if ($request->has('search') && ! empty($request->search['value'])) {
                $search = $request->search['value'];

                $records->where(function ($q) use ($search) {
                    $q->where('item_name', 'like', "%{$search}%")
                        ->orWhere('short_name', 'like', "%{$search}%")
                        ->orWhere('billing_name', 'like', "%{$search}%")
                        ->orWhere('unit_name', 'like', "%{$search}%")
                        ->orWhere('hsn_code', 'like', "%{$search}%")
                        ->orWhere('purchase_rate', 'like', "%{$search}%")
                        ->orWhere('opening_rate', 'like', "%{$search}%")
                        ->orWhereHas('category', fn ($qq) => $qq->where('name', 'like', "%{$search}%"))
                        ->orWhereHas('purchaseGroup', fn ($qq) => $qq->where('name', 'like', "%{$search}%"));
                });
            }

            /*if ($request->firm_id) {
                $records->where('firm_id', $request->firm_id);
            }*/

            if ($request->category_id) {
                $records->where('category_id', $request->category_id);
            }

            if ($request->purchase_group_id) {
                $records->where('purchase_group_id', $request->purchase_group_id);
            }

            if ($request->inclusive_tax) {
                $records->where('inclusive_tax', $request->inclusive_tax);
            }

            if ($request->start_date) {
                $records->whereDate('created_at', '>=', date('Y-m-d', strtotime($request->start_date)));
            }

            if ($request->end_date) {
                $records->whereDate('created_at', '<=', date('Y-m-d', strtotime($request->end_date)));
            }

            return dataTables()->of($records)
                ->addIndexColumn()

                // Merge Item Name and Short Name
                ->addColumn('item_details', function ($r) {
                    return '<div style="text-align:left;">
                                <a type="button" data-id="'.$r->id.'"
                                    data-bs-toggle="tooltip" title="Edit" onclick="editmodal('.$r->id.');"><strong>'.($r->item_name ?? '-').'</strong></a> <br>
                                <small class="text-muted">'.($r->short_name ?? '-').'</small>
                            </div>';
                })

                // Merge Purchase Rate, Opening Rate, Cost Rate as "Rate"
                ->addColumn('rates', function ($r) {
                    return '<div style="text-align:center;font-size:13px;">
                                <div><strong>Purchase:</strong> '.number_format($r->purchase_rate, 2).'</div>
                                <div><strong>Opening:</strong> '.number_format($r->opening_rate, 2).'</div>
                                <div><strong>Cost:</strong> '.number_format($r->cost_rate, 2).'</div>
                            </div>';
                })

                // Merge SGST, CGST, IGST, CESS as "GST"
                ->addColumn('gst_details', function ($r) {
                    return '<div style="text-align:center;font-size:13px;">
                                <div><strong>SGST:</strong> '.($r->sgst ? number_format($r->sgst, 2).'%' : '-').'</div>
                                <div><strong>CGST:</strong> '.($r->cgst ? number_format($r->cgst, 2).'%' : '-').'</div>
                                <div><strong>IGST:</strong> '.($r->igst ? number_format($r->igst, 2).'%' : '-').'</div>
                                <div><strong>CESS:</strong> '.($r->cess ? number_format($r->cess, 2).'%' : '-').'</div>
                            </div>';
                })
                ->addColumn('quantity', function ($r) {
                    return '<div style="text-align:center;font-size:13px;">
                                <div><strong>Max:</strong> '.($r->maximum_qty ? number_format($r->maximum_qty, 2) : '-').'</div>
                                <div><strong>Min:</strong> '.($r->minimum_qty ? number_format($r->minimum_qty, 2) : '-').'</div>
                            </div>';
                })

                ->editColumn('category', fn ($r) => $r->category?->name ?? '-')
                ->editColumn('purchaseGroup', fn ($r) => $r->purchaseGroup?->name ?? '-')
                ->editColumn('created_at', fn ($r) => $r->created_at?->format('d-m-Y'))

                ->addColumn('action', function ($record) {
                    $html = '<div style="display:flex;gap:5px;align-items:center;justify-content:center;">';

                    if (rolemanagement('R', 3)) {
                        $html .= '<button class="btn btn-sm btn-warning edit-btn"
                                    data-id="'.$record->id.'"
                                    data-bs-toggle="tooltip" title="Edit" onclick="editmodal('.$record->id.');">
                                    <i class="fas fa-edit"></i>
                                </button>';
                    }

                    if (rolemanagement('Product', 4)) {
                        $html .= '<button class="btn btn-sm btn-danger data-delete"
                                    data-id="'.$record->id.'"
                                    data-bs-toggle="tooltip" title="Delete">
                                    <i class="fas fa-trash"></i>
                                </button>';
                    }

                    $html .= '</div>';

                    return $html;
                })
                ->rawColumns(['item_details', 'rates', 'gst_details', 'category', 'purchaseGroup', 'quantity', 'action'])
                ->make(true);
        }

        return view('admin.raw_material.index', $this->data);
    }

    public function create()
    {
        if (! rolemanagement('Product', 2)) {
            return response()->json(['code' => -1, 'message' => 'Unauthorized'], 403);
        }

        $html = view('admin.raw_material.form')->render();

        return response()->json(['success' => true, 'html' => $html]);
    }

    public function store(Request $request)
    {
        if (! rolemanagement('Product', 2)) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $validated = $request->validate([
            /* 'firm'              => 'required|integer|exists:firms,id', */
            'item_name' => 'required|string|max:255',
            'short_name' => 'nullable|string|max:100',
            'billing_name' => 'nullable|string|max:255',
            'unit_name' => 'required|string|max:50',
            'purchase_rate' => 'required|numeric',
            'opening_rate' => 'required|numeric',
            'category' => 'required|integer|exists:categories,id',
            'group' => 'required|integer|exists:purchase_groups,id',
            'inclusive_tax1' => 'required|in:Yes,No',
            'sgst' => 'nullable|numeric',
            'cgst' => 'nullable|numeric',
            'igst' => 'nullable|numeric',
            'cess' => 'nullable|numeric',
            'hsn_code' => 'nullable|string|max:20',
            'maximum_qty' => 'nullable|numeric',
            'minimum_qty' => 'nullable|numeric',
            'reorder_level' => 'nullable|numeric',
            'reorder_qty' => 'nullable|numeric',
            'cost_rate' => 'nullable|numeric',
            'rr_item' => 'nullable|string|max:255',
            'receipt_item' => 'nullable|string|max:255',
        ]);

        // if ($validator->fails()) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'Please fix the highlighted errors and try again.',
        //         'errors'  => $validator->errors(),
        //     ], 422);
        // }

        $data = [
            /* 'firm_id'           => $validated['firm'], */
            'item_name' => $validated['item_name'],
            'short_name' => $validated['short_name'] ?? null,
            'billing_name' => $validated['billing_name'] ?? null,
            'unit_name' => $validated['unit_name'],
            'purchase_rate' => $validated['purchase_rate'],
            'opening_rate' => $validated['opening_rate'],
            'category_id' => $validated['category'] ?? null,
            'purchase_group_id' => $validated['group'] ?? null,
            'inclusive_tax' => $validated['inclusive_tax1'],
            'sgst' => $validated['sgst'] ?? null,
            'cgst' => $validated['cgst'] ?? null,
            'igst' => $validated['igst'] ?? null,
            'cess' => $validated['cess'] ?? null,
            'hsn_code' => $validated['hsn_code'] ?? null,
            'maximum_qty' => $validated['maximum_qty'] ?? 0,
            'minimum_qty' => $validated['minimum_qty'] ?? 0,
            'reorder_level' => $validated['reorder_level'] ?? 0,
            'reorder_qty' => $validated['reorder_qty'] ?? 0,
            'cost_rate' => $validated['cost_rate'] ?? null,
            'rr_item' => $validated['rr_item'] ?? null,
            'receipt_item' => $validated['receipt_item'] ?? null,
        ];

        RawMaterial::create($data);

        return response()->json(['success' => true, 'message' => 'Product has been added!']);
    }

    public function edit(Request $request)
    {
        if (! rolemanagement('Product', 3)) {
            return response()->json(['code' => -1, 'message' => 'Unauthorized'], 403);
        }

        $product = RawMaterial::find($request->id);
        if (! $product) {
            return response()->json(['success' => false, 'message' => 'Product not found.'], 404);
        }

        $html = view('admin.raw_material.form', compact('product'))->render();

        return response()->json(['success' => true, 'html' => $html]);
    }

    public function update(Request $request, $id)
    {
        if (! rolemanagement('Product', 3)) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $product = RawMaterial::findOrFail($id);

        $validated = $request->validate([
            /* 'firm'              => 'required|integer|exists:firms,id', */
            'item_name' => 'required|string|max:255',
            'short_name' => 'nullable|string|max:100',
            'billing_name' => 'nullable|string|max:255',
            'unit_name' => 'required|string|max:50',
            'purchase_rate' => 'required|numeric',
            'opening_rate' => 'required|numeric',
            'cost_rate' => 'nullable|numeric',
            'category' => 'nullable|integer|exists:categories,id',
            'group' => 'nullable|integer|exists:purchase_groups,id',
            'inclusive_tax1' => 'required|in:Yes,No',
            'sgst' => 'nullable|numeric',
            'cgst' => 'nullable|numeric',
            'igst' => 'nullable|numeric',
            'cess' => 'nullable|numeric',
            'hsn_code' => 'nullable|string|max:20',
            'maximum_qty' => 'nullable|numeric',
            'minimum_qty' => 'nullable|numeric',
            'reorder_level' => 'nullable|numeric',
            'reorder_qty' => 'nullable|numeric',
            'rr_item' => 'nullable|string|max:255',
            'receipt_item' => 'nullable|string|max:255',
        ]);

        // if ($validator->fails()) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'Validation failed. Please check your inputs.',
        //         'errors' => $validator->errors(),
        //     ], 422);
        // }

        $product->update([
            /* 'firm_id'           => $validated['firm'], */
            'item_name' => $validated['item_name'],
            'short_name' => $validated['short_name'] ?? null,
            'billing_name' => $validated['billing_name'] ?? null,
            'unit_name' => $validated['unit_name'],
            'purchase_rate' => $validated['purchase_rate'],
            'opening_rate' => $validated['opening_rate'],
            'cost_rate' => $validated['cost_rate'] ?? null,
            'category_id' => $validated['category'] ?? null,
            'purchase_group_id' => $validated['group'] ?? null,
            'inclusive_tax' => $validated['inclusive_tax1'],
            'sgst' => $validated['sgst'] ?? null,
            'cgst' => $validated['cgst'] ?? null,
            'igst' => $validated['igst'] ?? null,
            'cess' => $validated['cess'] ?? null,
            'hsn_code' => $validated['hsn_code'] ?? null,
            'maximum_qty' => $validated['maximum_qty'] ?? 0,
            'minimum_qty' => $validated['minimum_qty'] ?? 0,
            'reorder_level' => $validated['reorder_level'] ?? 0,
            'reorder_qty' => $validated['reorder_qty'] ?? 0,
            'rr_item' => $validated['rr_item'] ?? null,
            'receipt_item' => $validated['receipt_item'] ?? null,
        ]);

        return response()->json(['success' => true, 'message' => 'Product details have been updated!']);
    }

    public function destroy($id)
    {
        $product = RawMaterial::find($id);

        if (! $product) {
            return response()->json(['success' => false, 'message' => 'Product not found.'], 404);
        }

        $product->delete();

        return response()->json(['success' => true, 'message' => 'Product has been deleted!']);
    }
}
