<?php

namespace App\Http\Controllers\AdminAuth;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\PurchaseData\PurchaseGroup;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Str;


class PurchaseGroupController extends Controller
{
    private $data = [
        'route' => 'admin.purchasegroup.',
        'title' => 'Purchase Group',
        'menu' => 'Purchase Group',
        'submenu' => '',
    ];

    public function index(Request $request)
    {
        if (! rolemanagement('Purchase Group', 1)) {
            abort(401);
        }

        if ($request->ajax()) {
            $purchasegroup = $request->name;
            $startDate = $request->start_date;
            $endDate = $request->end_date;
            $status = $request->status;
            $categoryId = $request->category_id;

            $records = PurchaseGroup::with('category')
                ->orderBy('purchase_groups.id', 'desc');

            if ($purchasegroup) {
                $records->where('purchase_groups.name', 'like', "%{$purchasegroup}%");
            }

            if ($categoryId) {
                $records->where('purchase_groups.category_id', $categoryId);
            }

            if ($startDate) {
                $records->whereDate('purchase_groups.created_at', '>=', date('Y-m-d', strtotime($startDate)));
            }

            if ($endDate) {
                $records->whereDate('purchase_groups.created_at', '<=', date('Y-m-d', strtotime($endDate)));
            }

            if ($status) {
                $records->where('purchase_groups.status', $status);
            }

            return dataTables()->of($records)
                ->addIndexColumn()
                ->addColumn('category_name', fn ($record) => $record->category ? e($record->category->name) : '-')
                ->editColumn('status', function ($record) {
                    return '
                        <div class="form-check form-switch">
                            <input class="form-check-input status-toggle" id="statusToggle" type="checkbox" data-id="'.$record->id.'" data-status="'.$record->status.'" '.($record->status == 'Active' ? 'checked' : '').'>
                        </div>';
                })
                ->addColumn('name', function ($record) {
                    return '<a type="button" class="text-primary" onclick="catedit('.$record->id.')" 
                            data-bs-toggle="tooltip" 
                            title="Edit">
                            '.$record->name.'
                            </a>';
                })
                ->editColumn('created_at', fn ($record) => date('d-m-Y H:i A', strtotime($record->created_at)))
                ->editColumn('updated_at', fn ($record) => date('d-m-Y H:i A', strtotime($record->updated_at)))
                ->addColumn('action', function ($record) {
                    $html = '<div class="d-flex justify-content-center align-items-center gap-1">';
                    if (rolemanagement('Purchase Group', 3)) {
                        $html .= '<a type="button" onclick="catedit('.$record->id.')" class="btn btn-sm btn-info btn-b" title="Edit"><i class="fas fa-edit"></i></a>';
                    }
                    if (rolemanagement('Purchase Group', 4)) {
                        $html .= '<button class="btn btn-sm btn-danger data-delete" data-id="'.$record->id.'" title="Delete"><i class="fas fa-trash"></i></button>';
                    }
                    $html .= '</div>';

                    return $html;
                })
                ->rawColumns(['name', 'status', 'action'])
                ->make(true);
        }

        $this->data['categories'] = Category::where('status', 'Active')->orderBy('name', 'ASC')->get();

        return view('admin.purchasegroup.index', $this->data);
    }

    public function create()
    {
        if (! rolemanagement('Purchase Group', 2)) {
            abort(401);
        }

        $categories = Category::where('status', 'Active')->get();

        $categoryOptions = '<option value="">Select Category</option>';
        foreach ($categories as $cat) {
            $categoryOptions .= '<option value="'.$cat->id.'">'.e($cat->name).'</option>';
        }

        $html = '
        <form id="purchaseGroupForm">
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label>Category <span class="text-danger">*</span></label>
                        <select name="category_id" id="category_id" class="form-control">
                            '.$categoryOptions.'
                        </select>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label>Purchase Group Name <span class="text-danger">*</span></label>
                        <input type="text" name="name" class="form-control" id="name" placeholder="Enter purchase group name">
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="submit" id="submit" class="btn btn-primary">Submit</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal();">Close</button>
            </div>
        </form>';

        return response()->json(['success' => true, 'html' => $html]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'category_id' => 'required|exists:categories,id',
            'name' => 'required|string|max:255|unique:purchase_groups,name',
        ], [
            'category_id.required' => 'Please select a category.',
            'category_id.exists' => 'Selected category does not exist.',
            'name.required' => 'Please enter a purchase group name.',
            'name.unique' => 'This purchase group name already exists.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ]);
        }

        $record = new PurchaseGroup;
        $record->category_id = $request->category_id;
        $record->name = $request->name;
        $record->slug = Str::slug($request->name);
        $record->status = 'Active';
        $record->save();

        return response()->json(['success' => true, 'message' => 'Purchase Group added successfully.']);
    }

    public function edit(Request $request)
    {
        if (! rolemanagement('Purchase Group', 3)) {
            abort(401);
        }

        $record = PurchaseGroup::find($request->id);
        if (! $record) {
            return response()->json(['success' => false, 'message' => 'Purchase Group not found.']);
        }

        $categories = Category::where('status', 'Active')->get();

        $categoryOptions = '<option value="">Select Category</option>';
        foreach ($categories as $cat) {
            $selected = ($record->category_id == $cat->id) ? 'selected' : '';
            $categoryOptions .= '<option value="'.$cat->id.'" '.$selected.'>'.e($cat->name).'</option>';
        }

        $html = '
        <form id="purchaseGroupEditForm">
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label>Category <span class="text-danger">*</span></label>
                        <select name="category" id="category" class="form-control">
                            '.$categoryOptions.'
                        </select>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label>Purchase Group Name <span class="text-danger">*</span></label>
                        <input type="text" name="purchase_group_name" class="form-control" id="purchase_group_name" value="'.e($record->name).'">
                        <input type="hidden" name="id" id="id" value="'.e($record->id).'">
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="submit" id="submit" class="btn btn-primary">Update</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal();">Close</button>
            </div>
        </form>';

        return response()->json(['success' => true, 'html' => $html]);
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:purchase_groups,id',
            'category' => 'required|exists:categories,id',
            'purchase_group_name' => 'required|string|max:255|unique:purchase_groups,name,'.$request->id,
        ], [
            'category.required' => 'Please select a category.',
            'category.exists' => 'Selected category does not exist.',
            'purchase_group_name.required' => 'Please enter a purchase group name.',
            'purchase_group_name.unique' => 'This purchase group name already exists.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ]);
        }

        $record = PurchaseGroup::find($request->id);
        $record->category_id = $request->category;
        $record->name = $request->purchase_group_name;
        $record->slug = Str::slug($request->name);
        $record->save();

        return response()->json(['success' => true, 'message' => 'Purchase Group updated successfully.']);
    }

    public function sequence_update(Request $request)
    {
        $record = PurchaseGroup::find($request->id);
        if ($request->ajax() && $record) {
            $record->status = ($record->status == 'Active') ? 'Inactive' : 'Active';
            $record->save();

            return response()->json(['success' => true, 'message' => 'Status updated successfully.']);
        }

        return response()->json(['success' => false, 'message' => 'Record not found.']);
    }

    public function destroy(Request $request)
    {
        if (! rolemanagement('Purchase Group', 4)) {
            abort(401);
        }

        $record = PurchaseGroup::find($request->id);
        if ($record) {
            $record->delete();

            return response()->json(['success' => true, 'message' => 'Purchase Group deleted successfully.']);
        }

        return response()->json(['success' => false, 'message' => 'Record not found.']);
    }

    public function getPurchaseGroups($categoryId)
    {
        $groups = PurchaseGroup::where('category_id', $categoryId)->where('status', 'Active')->get(['id', 'name']);

        return response()->json($groups);
    }
}
