<?php

namespace App\Http\Controllers\AdminAuth;

use App\Http\Controllers\Controller;
use App\Models\PurchaseData\PurchaseOrder;
use App\Models\PurchaseData\PurchaseOrderItem;
use App\Models\RawMaterial;
use App\Models\Sitesetting;
use App\Models\VendorData\Vendor;
use Illuminate\Http\Request;

class PurchaseController extends Controller
{
    public function __construct() {}

    private $data = [
        'route' => 'admin.purchase.',
        'title' => 'Purchase',
        'menu' => 'Purchase',
        'submenu' => '',
    ];

    public function index(Request $request)
    {
        // If AJAX request → return datatable JSON
        if ($request->ajax()) {

            $query = PurchaseOrder::with(['vendor', 'items'])
                ->orderBy('id', 'DESC');

            // Filter : Vendor
            if (! empty($request->vendor_id)) {
                $query->where('vendor_id', $request->vendor_id);
            }

            // Filter : Date Range
            if (! empty($request->start_date)) {
                $query->whereDate('voucher_date', '>=', _d($request->start_date));
            }

            if (! empty($request->end_date)) {
                $query->whereDate('voucher_date', '<=', _d($request->end_date));
            }

            return datatables()->of($query)
                ->addIndexColumn()

                // Vendor Name
                ->addColumn('vendor_name', function ($row) {
                    return $row->vendor->name ?? '-';
                })

                // Voucher Date
                ->addColumn('voucher_date', function ($row) {
                    return _d($row->voucher_date);
                })

                // Total Amount
                ->addColumn('grand_total', function ($row) {
                    return number_format($row->grand_total, 2);
                })

                // Count Items
                ->addColumn('item_count', function ($row) {
                    return $row->items->count();
                })

                ->editColumn('created_at', function ($row) {
                    return $row->created_at->format('d-m-Y');
                })

                // Action Buttons
                ->addColumn('action', function ($row) {
                    $edit = '<a href="'.route('admin.purchase.edit', $row->id).'" class="btn btn-sm btn-info"><i class="fas fa-edit"></i></a>';

                    $delete = '<button class="btn btn-sm btn-danger data-delete"
                                    data-id="'.$row->id.'"><i class="fas fa-trash"></i></button>';

                    return $edit.' '.$delete;
                })

                ->rawColumns(['action'])
                ->make(true);
        }

        // Normal page load
        return view('admin.purchase.index');
    }

    public function create()
    {
        $voucher_no_int = PurchaseOrder::max('voucher_no_int');
        $voucher_no_int = $voucher_no_int ? $voucher_no_int + 1 : 1;

        $sitesetting = Sitesetting::first();
        $voucher_no_prefix = $sitesetting->voucher_no_prefix;
        $voucher_no = $voucher_no_prefix.str_pad($voucher_no_int, 5, '0', STR_PAD_LEFT);

        $this->data['voucher_no_int'] = $voucher_no_int;
        $this->data['voucher_no_prefix'] = $voucher_no_prefix;
        $this->data['voucher_no'] = $voucher_no;
        $this->data['vendors'] = Vendor::get();
        $this->data['rawMaterials'] = RawMaterial::get();

        return view('admin.purchase.create', $this->data);
    }

    public function store(Request $request)
    {

        $vendor = Vendor::find($request->vendor_id);

        $order = PurchaseOrder::create([

            'user_id' => auth('admin')->id(),
            'vendor_id' => $request->vendor_id,

            'party_name' => $vendor->name,

            'gst_name' => $request->billing_name,

            'voucher_no' => $request->voucher_no,
            'voucher_date' => sql_date($request->voucher_date),
            'party_ref_no' => $request->party_ref_no,
            'ref_date' => sql_date($request->ref_date),
            'po_no' => $request->po_no,
            'po_date' => sql_date($request->po_date),
            'payment_dr_cr' => $request->payment_dr_cr,
            'invoice' => $request->invoice,
            'gst_rcm_itc' => $request->gst_rcm_itc,
            'rcm' => $request->rcm,
            'total_discount' => $request->total_discount,
            'subtotal' => $request->subtotal,
            'tax_amount' => $request->tax_amount,
            'grand_total' => $request->grand_total,
            'payable_amount' => $request->payable_amount,

            'total_cess' => $request->total_cess,
            'total_cgst' => $request->total_cgst,
            'total_sgst' => $request->total_sgst,
            'total_igst' => $request->total_igst,

        ]);

        foreach ($request->items as $item) {

            $order->items()->create([

                'p' => $item['p'],
                'item_id' => $item['item_id'] ?? null,
                'item_name' => $item['item_name'] ?? null,
                'pack' => $item['pack'],
                'grn_no' => $item['grn_no'],
                'quantity' => $item['quantity'],
                'net_rate' => $item['net_rate'],
                'discount' => $item['discount'] ?? 0,
                'amount' => $item['amount'],
                'cgst' => $item['cgst'] ?? 0,
                'sgst' => $item['sgst'] ?? 0,
                'igst' => $item['igst'] ?? 0,
                'cess' => $item['cess'] ?? 0,
                'tax_amount' => $item['tax_amount'] ?? 0,

            ]);

        }

        return $this->edit($order->id);

    }

    public function edit($id)
    {

        $vendors = Vendor::get();
        $rawMaterials = RawMaterial::get();
        $purchase = PurchaseOrder::with('items')->find($id);

        return view('admin.purchase.create', compact('purchase', 'vendors', 'rawMaterials'));

    }

    public function update(Request $request, $id)
    {

        $purchase = PurchaseOrder::findOrFail($id);

        $vendor = Vendor::find($request->vendor_id);

        // --- Update main purchase ---
        $purchase->update([
            'vendor_id' => $vendor->id,
            'party_name' => $vendor->name,
            'gst_name' => $request->gst_name,

            'voucher_no' => $request->voucher_no,
            'voucher_no_int' => $request->voucher_no_int,
            'voucher_date' => sql_date($request->voucher_date),

            'party_ref_no' => $request->party_ref_no,
            'ref_date' => sql_date($request->ref_date),
            'po_no' => $request->po_no,
            'po_date' => sql_date($request->po_date),

            'payment_dr_cr' => $request->payment_dr_cr,
            'invoice' => $request->invoice,
            'gst_rcm_itc' => $request->gst_rcm_itc,
            'rcm' => $request->rcm ? 1 : 0,

            // --- Totals coming from hidden fields ---
            'total_discount' => $request->total_discount,
            'subtotal' => $request->subtotal,
            'tax_amount' => $request->tax_amount,
            'grand_total' => $request->grand_total,
        ]);

        // Insert new items
        if ($request->items) {
            foreach ($request->items as $item) {
                if (! empty($item['id'])) {

                    $purchaseItem = PurchaseOrderItem::find($item['id']);

                    if ($purchaseItem) {
                        $purchaseItem->update([
                            'p' => $item['p'],
                            'item_id' => $item['item_id'] ?? null,
                            'item_name' => $item['item_name'] ?? null,
                            'pack' => $item['pack'],
                            'grn_no' => $item['grn_no'],
                            'quantity' => $item['quantity'],
                            'net_rate' => $item['net_rate'],
                            'discount' => $item['discount'] ?? 0,
                            'amount' => $item['amount'],
                            'cgst' => $item['cgst'] ?? 0,
                            'sgst' => $item['sgst'] ?? 0,
                            'igst' => $item['igst'] ?? 0,
                            'cess' => $item['cess'] ?? 0,
                            'tax_amount' => $item['tax_amount'] ?? 0,
                        ]);

                        $updatedItemIds[] = $purchaseItem->id;
                    }

                } else {
                    // If ID missing → This is NEW ROW → INSERT
                    $new = $purchase->items()->create([
                        'p' => $item['p'],
                        'item_id' => $item['item_id'] ?? null,
                        'item_name' => $item['item_name'] ?? null,
                        'pack' => $item['pack'],
                        'grn_no' => $item['grn_no'],
                        'quantity' => $item['quantity'],
                        'net_rate' => $item['net_rate'],
                        'discount' => $item['discount'] ?? 0,
                        'amount' => $item['amount'],
                        'cgst' => $item['cgst'] ?? 0,
                        'sgst' => $item['sgst'] ?? 0,
                        'igst' => $item['igst'] ?? 0,
                        'cess' => $item['cess'] ?? 0,
                        'tax_amount' => $item['tax_amount'] ?? 0,
                    ]);

                    $updatedItemIds[] = $new->id;
                }
            }

            // Delete old items
            $purchase->items()->whereNotIn('id', $updatedItemIds)->delete();

        }

        return redirect()
            ->route('admin.purchase.index')
            ->with('success', 'Purchase Updated Successfully!');
    }
}
