<?php

namespace App\Http\Controllers\AdminAuth;

use App\Http\Controllers\Controller;
use App\Models\City;
use App\Models\Firm;
use App\Models\State;
use DB;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Yajra\DataTables\Facades\DataTables;

class FirmController extends Controller
{
    private $data = [
        'route' => 'admin.firms.',
        'title' => 'Firms',
        'menu' => 'Firms',
        'submenu' => '',
    ];

    public function index(Request $request)
    {
        if (! rolemanagement('Firm', 1)) {
            abort(401);
        }

        if ($request->ajax()) {
            $records = Firm::orderBy('id', 'DESC')
                ->with(['stateInfo', 'cityInfo'])
                ->select('firms.*');

            $records = restrictByAuthor($records);

            if ($request->has('search') && ! empty($request->search['value'])) {
                $search = $request->search['value'];
                $records->where(function ($q) use ($search) {
                    $q->where('firm_name', 'like', "%{$search}%")
                        ->orWhere('owner_name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('mobile_number', 'like', "%{$search}%")
                        ->orWhere('gst_number', 'like', "%{$search}%")
                        ->orWhere('pan_number', 'like', "%{$search}%")
                        ->orWhere('aadhar_number', 'like', "%{$search}%")
                        ->orWhereHas('stateInfo', function ($query) use ($search) {
                            $query->where('name', 'like', "%{$search}%");
                        })
                        ->orWhereHas('cityInfo', function ($query) use ($search) {
                            $query->where('name', 'like', "%{$search}%");
                        });
                });
            }

            return dataTables()->of($records)
                ->addIndexColumn()
                ->editColumn('state', fn ($r) => $r->stateInfo->name ?? '-')
                ->editColumn('city', fn ($r) => $r->cityInfo->name ?? '-')
                ->editColumn('created_at', fn ($r) => date('d-m-Y', strtotime($r->created_at)))
                ->editColumn('status', function ($record) {
                    return '<div class="form-check form-switch">
                                <input class="form-check-input status-toggle pointer"
                                       type="checkbox" data-id="'.$record->id.'"
                                       '.($record->status === 'Active' ? 'checked' : '').'>
                            </div>';
                })
                ->addColumn('action', function ($record) {
                    $html = '<div style="display: flex; gap: 5px; align-items: center; justify-content: center;">';
                    if (rolemanagement('Firm', 3)) {
                        $html .= '<a href="'.route('admin.firms.edit', $record->id).'"
                                    class="btn btn-sm btn-info" data-bs-toggle="tooltip" title="Edit">
                                    <i class="fas fa-edit"></i>
                                  </a>';
                    }
                    if (rolemanagement('Firm', 4)) {
                        $html .= '<button class="btn btn-sm btn-danger data-delete"
                                    data-id="'.$record->id.'" data-bs-toggle="tooltip" title="Delete">
                                    <i class="fas fa-trash"></i>
                                  </button>';
                    }
                    $html .= '</div>';

                    return $html;
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
        }

        return view('admin.firms.index', $this->data);
    }

    public function create()
    {
        if (! rolemanagement('Firm', 2)) {
            abort(401);
        }

        $firm = new Firm;
        $firm->id = null;

        $this->data['firm'] = $firm;
        $this->data['states'] = State::where('status', 'Active')->orderBy('name', 'asc')->get();
        $this->data['cities'] = City::where('status', 'Active')->orderBy('name', 'asc')->get();

        return view('admin.firms.create', $this->data);
    }

    public function store(Request $request)
    {
        $rules = $this->validationRules();

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            $data = $this->prepareData($request);
            $data['author_id'] = auth('admin')->user()->role_id != 1 ? auth('admin')->id() : null;

            $firm = Firm::create($data);
            $firm->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Firm added successfully!',
                'redirect_url' => route('admin.firms.index'),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error: '.$e->getMessage(),
            ], 500);
        }
    }

    public function edit($id)
    {
        if (! rolemanagement('Firm', 3)) {
            abort(401);
        }

        $this->data['firm'] = restrictByAuthor(Firm::where('id', $id))->firstOrFail();
        $this->data['states'] = State::where('status', 'Active')->orderBy('name', 'asc')->get();
        $this->data['cities'] = City::where('status', 'Active')->orderBy('name', 'asc')->get();

        return view('admin.firms.create', $this->data);
    }

    public function update(Request $request, $id)
    {
        $firm = restrictByAuthor(Firm::findOrFail($id));

        $rules = $this->validationRules($id);

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            $data = $this->prepareData($request, $firm);

            $firm->update($data);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Firm updated successfully!',
                'redirect_url' => route('admin.firms.index'),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error: '.$e->getMessage(),
            ], 500);
        }
    }

    public function destroy(Request $request)
    {
        if (! rolemanagement('Firm', 4)) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $firm = restrictByAuthor(Firm::where('id', $request->id))->first();

        if (! $firm) {
            return response()->json(['success' => false, 'message' => 'Firm not found']);
        }

        DB::beginTransaction();
        try {
            $this->deleteFileIfExists($firm->logo);
            $this->deleteFileIfExists($firm->pan_card_image);
            $this->deleteFileIfExists($firm->aadhar_card_image);
            $this->deleteFileIfExists($firm->gst_certificate);
            $this->deleteFileIfExists($firm->bank_passbook_image);
            $this->deleteFileIfExists($firm->registration_certificate);

            $firm->delete();
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Firm deleted successfully.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error: '.$e->getMessage(),
            ]);
        }
    }

    public function toggleStatus(Request $request, $id)
    {
        $firm = restrictByAuthor(Firm::where('id', $id))->firstOrFail();
        $firm->status = $firm->status === 'Active' ? 'Inactive' : 'Active';
        $firm->save();

        return response()->json([
            'success' => true,
            'message' => 'Status updated successfully!',
            'status' => $firm->status,
        ]);
    }

    private function validationRules($id = null)
    {
        $uniqueEmail = Rule::unique('firms', 'email')->ignore($id);
        $uniqueMobile = Rule::unique('firms', 'mobile_number')->ignore($id);
        $uniquePan = Rule::unique('firms', 'pan_number')->ignore($id);
        $uniqueAadhar = Rule::unique('firms', 'aadhar_number')->ignore($id);
        $uniqueGST = Rule::unique('firms', 'gst_number')->ignore($id);
        $uniqueRegNo = Rule::unique('firms', 'registration_number')->ignore($id);
        $uniqueAccount = Rule::unique('firms', 'account_number')->ignore($id);
        $uniqueIfsc = Rule::unique('firms', 'ifsc_code')->ignore($id);
        $uniqueUpi = Rule::unique('firms', 'upi_id')->ignore($id);
        $uniqueTan = Rule::unique('firms', 'tan_number')->ignore($id);
        $uniqueCin = Rule::unique('firms', 'cin_number')->ignore($id);
        $uniqueMsme = Rule::unique('firms', 'msme_number')->ignore($id);

        return [
            'firm_name' => 'required|string|max:255',
            'firm_type' => 'required|string|max:255',
            'registration_number' => ['required', 'string', 'max:255', $uniqueRegNo],
            'owner_name' => 'required|string|max:255',
            'email' => ['required', 'email', 'max:255', $uniqueEmail],
            'mobile_number' => ['required', 'numeric', 'digits:10', $uniqueMobile],
            'gst_number' => ['required', 'string', 'regex:/^[0-9]{2}[A-Z]{5}[0-9]{4}[A-Z]{1}[1-9A-Z]{1}Z[0-9A-Z]{1}$/', $uniqueGST],
            'pan_number' => ['required', 'string', 'size:10', $uniquePan],
            'aadhar_number' => ['required', 'numeric', 'digits:12', $uniqueAadhar],
            'address_line1' => 'required|string|max:1000',
            'state' => 'required|exists:states,id',
            'city' => 'required|exists:cities,id',
            'status' => 'required',

            'date_of_establishment' => 'nullable|date',
            'designation' => 'nullable|string|max:255',
            'alternate_mobile_number' => 'nullable|numeric|digits:10',
            'address_line2' => 'nullable|string|max:1000',
            'district' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:255',
            'pincode' => 'nullable|numeric|digits:6',
            'bank_name' => 'nullable|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_holder_name' => 'nullable|string|max:255',
            'account_number' => ['nullable', 'string', 'max:255', $uniqueAccount],
            'ifsc_code' => ['nullable', 'string', 'regex:/^[A-Z]{4}0[A-Z0-9]{6}$/', $uniqueIfsc],
            'upi_id' => ['nullable', 'string', 'max:255', $uniqueUpi],
            'tan_number' => ['nullable', 'string', 'max:255', $uniqueTan],
            'cin_number' => ['nullable', 'string', 'max:255', $uniqueCin],
            'msme_number' => ['nullable', 'string', 'max:255', $uniqueMsme],
            'website_url' => 'nullable|url',
            'facebook_url' => 'nullable|url',
            'instagram_url' => 'nullable|url',
            'linkedin_url' => 'nullable|url',
            'remarks' => 'nullable|string|max:2000',

            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
            'pan_card_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
            'aadhar_card_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
            'gst_certificate' => 'nullable|mimes:jpeg,png,jpg,gif,pdf|max:5120',
            'bank_passbook_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
            'registration_certificate' => 'nullable|mimes:jpeg,png,jpg,gif,pdf|max:5120',
        ];
    }

    private function prepareData(Request $request, ?Firm $firm = null)
    {
        $data = [
            'firm_name' => $request->firm_name,
            'owner_name' => $request->owner_name,
            'email' => $request->email,
            'mobile_number' => $request->mobile_number,
            'gst_number' => $request->gst_number,
            'state' => $request->state,
            'city' => $request->city,
            'address_line1' => $request->address_line1,
            'status' => $request->status ?? 'Active',
            'firm_type' => $request->firm_type,
            'registration_number' => $request->registration_number,
            'designation' => $request->designation,
            'alternate_mobile_number' => $request->alternate_mobile_number,
            'pan_number' => $request->pan_number,
            'aadhar_number' => $request->aadhar_number,
            'tan_number' => $request->tan_number,
            'cin_number' => $request->cin_number,
            'msme_number' => $request->msme_number,
            'address_line2' => $request->address_line2,
            'district' => $request->district,
            'country' => $request->country ?? 'India',
            'pincode' => $request->pincode,
            'bank_name' => $request->bank_name,
            'branch_name' => $request->branch_name,
            'account_holder_name' => $request->account_holder_name,
            'account_number' => $request->account_number,
            'ifsc_code' => $request->ifsc_code,
            'upi_id' => $request->upi_id,
            'website_url' => $request->website_url,
            'facebook_url' => $request->facebook_url,
            'instagram_url' => $request->instagram_url,
            'linkedin_url' => $request->linkedin_url,
            'remarks' => $request->remarks,
        ];

        if (! empty($request->date_of_establishment)) {
            try {
                $data['date_of_establishment'] = \Carbon\Carbon::createFromFormat('d-m-Y', $request->date_of_establishment)
                    ->format('Y-m-d');
            } catch (\Exception $e) {
                $data['date_of_establishment'] = null;
            }
        } else {
            $data['date_of_establishment'] = null;
        }

        $basePath = 'uploads/firms';
        $files = [
            'logo',
            'pan_card_image',
            'aadhar_card_image',
            'gst_certificate',
            'bank_passbook_image',
            'registration_certificate',
        ];

        foreach ($files as $field) {
            if ($request->hasFile($field)) {
                if ($firm && $firm->$field) {
                    $this->deleteFileIfExists($firm->$field);
                }

                $file = $request->file($field);
                $uploadPath = $basePath.'/'.$field;
                if (! File::exists(public_path($uploadPath))) {
                    File::makeDirectory(public_path($uploadPath), 0755, true);
                }

                $filename = time().'_'.$file->getClientOriginalName();
                $file->move(public_path($uploadPath), $filename);
                $data[$field] = $uploadPath.'/'.$filename;
            }
        }

        return $data;
    }

    private function deleteFileIfExists($path)
    {
        if ($path && file_exists(public_path($path))) {
            unlink(public_path($path));
        }
    }
}
