<?php

namespace App\Http\Controllers\AdminAuth;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CategoryController extends Controller
{
    public function __construct() {}

    private $data = [
        'route' => 'admin.category.',
        'title' => 'Category',
        'menu' => 'Category',
        'submenu' => '',
    ];

    public function index(Request $request)
    {
        if (! rolemanagement('Category', 1)) {
            abort(401);
        }

        if ($request->ajax()) {
            $categoryName = $request->category_name;
            $startDate = $request->start_date;
            $endDate = $request->end_date;
            $status = $request->status;

            $records = Category::query()->orderBy('id', 'desc');

            if ($categoryName) {
                $records->where('name', 'like', "%{$categoryName}%");
            }

            if ($startDate) {
                $formattedStart = date('Y-m-d', strtotime($startDate));
                $records->whereDate('created_at', '>=', $formattedStart);
            }

            if ($endDate) {
                $formattedEnd = date('Y-m-d', strtotime($endDate));
                $records->whereDate('created_at', '<=', $formattedEnd);
            }

            if ($status) {
                $records->where('status', $status);
            }

            return dataTables()->of($records)
                ->addIndexColumn()
                ->editColumn('status', function ($record) {
                    return '
                        <div class="form-check form-switch">
                            <input class="form-check-input status-toggle"
                                   id="statusToggle"
                                   type="checkbox"
                                   data-id="'.$record->id.'"
                                   data-status="'.$record->status.'"
                                   '.($record->status == 'Active' ? 'checked' : '').'>
                        </div>';
                })
                ->editColumn('created_at', function ($record) {
                    return date('d-m-Y H:i A', strtotime($record->created_at));
                })
                ->editColumn('updated_at', function ($record) {
                    return date('d-m-Y H:i A', strtotime($record->updated_at));
                })
                ->addColumn('name', function ($record) {
                    return '<a type="button" class="text-primary" onclick="catedit('.$record->id.')" 
                               data-bs-toggle="tooltip" 
                               title="Edit">
                               '.$record->name.'
                            </a>';
                })
                ->addColumn('action', function ($record) {
                    $html = '<div class="d-flex justify-content-center align-items-center gap-1">';
                    if (rolemanagement('Category', 3)) {
                        $html .= '
                            <a type="button" onclick="catedit('.$record->id.')" 
                               class="btn btn-sm btn-info btn-b"
                               data-bs-toggle="tooltip" 
                               title="Edit">
                               <i class="fas fa-edit"></i>
                            </a>';
                    }
                    if (rolemanagement('Category', 4)) {
                        $html .= '
                            <button class="btn btn-sm btn-danger data-delete"
                                    data-id="'.$record->id.'"
                                    data-bs-toggle="tooltip"
                                    title="Delete">
                                <i class="fas fa-trash"></i>
                            </button>';
                    }
                    $html .= '</div>';

                    return $html;
                })
                ->rawColumns(['name', 'status', 'action'])
                ->make(true);
        }

        return view('admin.category.index', $this->data);
    }

    public function create()
    {
        if (! rolemanagement('Category', 2)) {
            abort(401);
        }

        $html = '
        <form id="catform">
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label for="">Name <span class="text-danger">*</span></label>
                        <input type="text" name="name" class="form-control" id="name">
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="submit" id="submit" class="btn btn-primary">Submit</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal();">Close</button>
            </div>
        </form>';

        return response()->json([
            'success' => true,
            'html' => $html,
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            $response['success'] = false;
            $response['message'] = $validator->errors();

            return response()->json($response);
        }

        $record = new Category;
        $record->name = $request->name;
        $record->status = 'Active';
        $record->save();

        $response['success'] = true;
        $response['message'] = 'Category has been added.';

        return response()->json($response);
    }

    public function edit(Request $request)
    {
        if (! rolemanagement('Category', 3)) {
            abort(401);
        }

        $record = Category::find($request->id);
        if (! $record) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found.',
            ]);
        }

        $html = '
        <form id="cateditform">
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label for="">Name <span class="text-danger">*</span></label>
                        <input type="text" name="name" class="form-control" id="name" value="'.e($record->name).'">
                        <input type="hidden" name="id" id="id" value="'.e($record->id).'">
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="submit" id="submit" class="btn btn-primary">Submit</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal();">Close</button>
            </div>
        </form>';

        return response()->json([
            'success' => true,
            'html' => $html,
        ]);
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
        ]);
        if ($validator->fails()) {
            $response['success'] = false;
            $response['message'] = $validator->errors();

            return response()->json($response);
        }

        $record = Category::find($request->id);
        $record->name = $request->name;
        $record->update();

        $response['success'] = true;
        $response['message'] = 'Update successfully';

        return response()->json($response);
    }

    public function sequence_update(Request $request)
    {
        $record = Category::find($request->id);
        if ($request->ajax()) {
            if (! empty($request->currentStatus)) {
                if ($request->currentStatus == 'Active') {
                    $record->status = 'Inactive';
                } else {
                    $record->status = 'Active';
                }
                $record->save();
                $response['success'] = true;
                $response['message'] = 'Update successfully';

                return response()->json($response);
            }
            $record->update();
            $response['success'] = true;
            $response['message'] = 'Update successfully';

            return response()->json($response);
        }
    }

    public function destroy(Request $request)
    {
        if (! rolemanagement('Category', 4)) {
            abort(401);
        }
        $record = Category::find($request->id);
        if ($record) {
            $record->delete();

            return response()->json([
                'success' => true,
                'message' => 'Category has been deleted',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Category not found',
        ]);
    }

    public function categorydelete($id)
    {
        Category::where('id', '=', $id)->delete();
        $response['success'] = true;
        $response['message'] = 'Category has been deleted';

        return response()->json($response);
    }
}
