<?php
namespace App\Exports;

use App\Models\CampaignDetail;
use App\Models\City;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithMapping;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class SuccessLeadExport implements FromCollection, WithHeadings, ShouldAutoSize, WithMapping
{
    protected $request;

    public function __construct($request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $memberId = Auth::guard('member')->id();

        $records = CampaignDetail::with(['database', 'member'])->whereHas('campaign', fn($q) => $q->where('member_id', $memberId))->where('status', 'Interested');

        $search = $this->request->input('search');
        if ($search) {
            $records->where(function($q) use($search){
                $q->orWhere(fn($q2) => $q2->where('status','Interested')->where('database_number','like',"%$search%"))
                  ->orWhereHas('database', function($q3) use($search) {
                      $q3->where('name','like',"%$search%");
                  });
            });
        }

        if ($this->request->filled('campaign_id')) {
            $records->where('campaign_id', $this->request->campaign_id);
        }

        if ($this->request->filled('city')) {
            $records->whereRaw("FIND_IN_SET(?, database_city_id)", [$this->request->city]);
        }

        if ($this->request->filled('subcategory')) {
            $subcategory = $this->request->subcategory;
            $records->whereHas('campaign', function ($q) use ($subcategory) {
                $q->whereNotNull('sub_category')
                  ->whereRaw("FIND_IN_SET(?, campaigns.sub_category)", [$subcategory]);
            });
        }

        if ($this->request->filled('language')) {
            $language = $this->request->language;
            $records->whereHas('campaign', function($q) use ($language) {
                $q->where('language', $language);
            });
        }

        if ($this->request->filled('startTime')) {
            $date = Carbon::createFromFormat('d-m-Y', $this->request->startTime)->format('Y-m-d');
            $records->whereDate('start_time', '>=', $date);
        }

        if ($this->request->filled('endTime')) {
            $date = Carbon::createFromFormat('d-m-Y', $this->request->endTime)->format('Y-m-d');
            $records->whereDate('end_time', '<=', $date);
        }

        return $records->orderBy('id', 'ASC')->get();
    }

    public function headings(): array
    {
        return ['Business Name','City','Number','Sell Price','Call Date','Start Time','End Time','Call Duration','Record','Status','Call Status'];
    }

    public function map($value): array
    {
        $city_name = City::whereIn('id', explode(',', $value->database->city ?? ''))->pluck('name')->implode(', ');
        $database_number = ($value->status === 'Interested') ? $value->database_number : maskMobile($value->database_number);
        $record_audio = $value->voice_record_url ? $value->voice_record_url : '-';

        return [
            $value->database->name ?? '-',         
            $city_name ?? '-',                     
            $database_number,         
            $value->sell_price ?? '-',             
            $value->call_date ? date('d-m-Y', strtotime($value->call_date)) : '-',  
            $value->start_time ? date('d-m-Y h:i:s A', strtotime($value->start_time)) : '-', 
            $value->end_time ? date('d-m-Y h:i:s A', strtotime($value->end_time)) : '-',    
            $value->call_duration ? secendConvertMin($value->call_duration) : '',       
            $record_audio,                                    
            $value->status ?? '-',                    
            $value->call_status ?? '-'        
        ];
    }
}
