<?php
namespace App\Exports;

use App\Models\CampaignDetail;
use App\Models\City;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithMapping;

class CampaignReportExportAdmin implements FromCollection, WithHeadings, ShouldAutoSize, WithMapping
{
    protected $request;
    protected $campaignId;

    public function __construct($request, $campaignId)
    {
        $this->request = $request;
        $this->campaignId = $campaignId;
    }

    public function collection()
    {
        $query = CampaignDetail::select('campaign_details.*')->with(['database', 'staff', 'member', 'reference', 'campaign'])->where('campaign_id', $this->campaignId)->orderBy('id', 'ASC');

        $search = $this->request->search_data ?? null;
        $query->when($search, function ($q) use ($search) {
            $q->where(function ($q) use ($search) {
                $q->where('status', 'like', "%$search%")
                  ->orWhereHas('staff', fn($q) => $q->where('name', 'like', "%$search%"))
                  ->orWhereHas('member', fn($q) => $q->where('name', 'like', "%$search%"))
                  ->orWhereHas('reference', fn($q) => $q->where('name', 'like', "%$search%"))
                  ->orWhereHas('campaign.product', fn($q) => $q->where('name', 'like', "%$search%"));
            });
        });

        $callStatus = $this->request->call_status ?? null;
        $query->when($callStatus, fn($q) => $q->whereRaw('LOWER(call_status) = ?', [strtolower($callStatus)]));

        $status = $this->request->status ?? null;
        $query->when($status, fn($q) => $q->whereRaw('LOWER(status) = ?', [strtolower($status)]));

        $startDate = $this->request->start_date ?? null;
        $query->when($startDate, fn($q) => $q->whereDate('start_time', '>=', date('Y-m-d', strtotime($startDate))));

        $endDate = $this->request->end_date ?? null;
        $query->when($endDate, fn($q) => $q->whereDate('end_time', '<=', date('Y-m-d', strtotime($endDate))));
        
        return $query->get();
    }

    public function headings(): array
    {
        return [
            'Business Name','City','Number','Product','Staff','Sell Price','Call Duration','Call Date','Start Time','End Time','Record','Reference By','Reference Commission (Paisa)','Commission Total','Status','Call Status','Telecaller Status'
        ];
    }

    public function map($record): array
    {
        $cityIds = $record->database->city ?? null;
        $cityName = $cityIds ? City::whereIn('id', explode(',', $cityIds))->pluck('name')->implode(', ') : '-';

        $databaseNumber = $record->status !== 'Interested' ? maskMobile($record->database_number) : $record->database_number;

        $commissionTotal = round($record->commission_total / 100, 2);

        $callDuration = $record->call_duration ? secendConvertMin($record->call_duration) : '';

        $timeSpent = '-';
        if (!empty($record->timeSpent) && !is_string($record->timeSpent)) {
            $parts = explode(':', $record->timeSpent);
            $seconds = ($parts[0] * 3600) + ($parts[1] * 60) + $parts[2];
            $timeSpent = secendConvertMin($seconds);
        }

        $recordHtml = $record->voice_record_url ? $record->voice_record_url : '-';

        return [
            $record->database->name ?? '-',
            $cityName,
            $databaseNumber,
            $record->campaign->product->name ?? '-',
            $record->staff->name ?? '-',
            $record->sell_price ?? '-',
            $callDuration,
            $record->call_date ? date('d-m-Y', strtotime($record->call_date)) : '-',
            $record->start_time ? date('d-m-Y h:i:s A', strtotime($record->start_time)) : '-',
            $record->end_time ? date('d-m-Y h:i:s A', strtotime($record->end_time)) : '-',
            $recordHtml,
            $record->reference->name ?? '-',
            $record->reference_commission ?? '-',
            $commissionTotal,
            $record->status ?? '-',
            $record->call_status ?? '-',
            $record->telecaller_status ?? '-'
        ];
    }
}
