<?php
namespace App\Exports;

use App\Models\CampaignDetail;
use App\Models\City;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use Illuminate\Support\Facades\Auth;

class CallDetailsExport implements FromCollection, WithHeadings, ShouldAutoSize, WithMapping, WithEvents
{
    protected $request;
    protected $totalWorkingEarn;
    protected $totalLeadEarn;

    public function __construct($request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $request = $this->request;
        $records = CampaignDetail::select('campaign_details.*')->where('product_id', 1)
            ->when(Auth::guard('admin')->user()->role_id != 1, function ($query) {
                return $query->where('staff_id', Auth::guard('admin')->user()->id);
            })
            ->with(['database', 'staff', 'member', 'reference', 'campaign'])
            ->orderBy('id', 'desc');
        
        $orderDetailSearch = $request->search_data;
        $records->when($orderDetailSearch, function ($query) use ($orderDetailSearch) {
            $query->where(function ($q) use ($orderDetailSearch) {
                $q->where('status', 'like', "%$orderDetailSearch%");
                $q->orWhereHas('staff', function ($q) use ($orderDetailSearch) {
                    $q->where('name', 'like', "%$orderDetailSearch%");
                });
                $q->orWhereHas('member', function ($q) use ($orderDetailSearch) {
                    $q->where('name', 'like', "%$orderDetailSearch%");
                });
                $q->orWhereHas('reference', function ($q) use ($orderDetailSearch) {
                    $q->where('name', 'like', "%$orderDetailSearch%");
                });
                $q->orWhereHas('campaign.product', function ($q) use ($orderDetailSearch) {
                    $q->where('name', 'like', "%$orderDetailSearch%");
                });
            });
        });

        $campaignStatus = $request->call_status;
        $records->when($campaignStatus, function ($query) use ($campaignStatus) {
            $query->whereRaw('LOWER(call_status) = ?', [strtolower($campaignStatus)]);
        });

        $status = $request->status;
        $records->when($status, function ($query) use ($status) {
            $query->whereRaw('LOWER(status) = ?', [strtolower($status)]);
        });

        $campaignStartDate = $request->start_date;
        $records->when($campaignStartDate, function ($query) use ($campaignStartDate) {
            $query->whereDate('start_time', '>=', date('Y-m-d', strtotime($campaignStartDate)));
        });

        $campaignEndDate = $request->end_date;
        $records->when($campaignEndDate, function ($query) use ($campaignEndDate) {
            $query->whereDate('end_time', '<=', date('Y-m-d', strtotime($campaignEndDate)));
        });

        $collection = $records->get();
        $this->totalWorkingEarn = $collection->sum('working_earn');
        $this->totalLeadEarn = $collection->sum('lead_earn');

        return $collection;
    }

    public function headings(): array
    {
        return ['Staff', 'Business Name', 'City', 'Number', 'Working Earn', 'Lead Earn', 'Call Date', 'Start Time', 'End Time', 'Time Spent', 'Call Duration', 'Record', 'Status', 'Call Status', 'Telecaller Status'];
    }

    public function map($value): array
    {
        $city = $value->database->city ?? null;
        if ($city) {
            $city_name = City::whereIn('id', explode(',', $city))->pluck('name')->implode(', ');
        } else {
            $city_name = '-';
        }
		$database_number = $value->database_number;
        if($value->status!='Interested'){
			$database_number = maskMobile($value->database_number);
		}
		
        if (!is_string($value->timeSpent) || empty($value->timeSpent)) {
            $timeSpent = '-';
        } else {
            $timeParts = explode(':', $value->timeSpent);
            $seconds = ($timeParts[0] * 3600) + ($timeParts[1] * 60) + $timeParts[2];
            $timeSpent = $seconds ? secendConvertMin($seconds) : '-';
        }
        return [
            $value->staff->name ?? '-',
            $value->database->name ?? '-',
            $city_name ?? '-',
            $database_number,
            $value->working_earn ?? '-',
            $value->lead_earn ?? '-',
            $value->call_date ? date('d-m-Y', strtotime($value->call_date)) : '-',
            $value->start_time ? date('d-m-Y h:i:s A', strtotime($value->start_time)) : '-',
            $value->end_time ? date('d-m-Y h:i:s A', strtotime($value->end_time)) : '-',
            $timeSpent,
            $value->call_duration ? secendConvertMin($value->call_duration) : '-',
            $value->record ?? '-',
            $value->status ?? '-',
            $value->call_status ?? '-',
            $value->telecaller_status ?? '-',
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $highestRow = $event->sheet->getHighestRow();
                $emptyRow = $highestRow + 1; 
                $totalRow = $highestRow + 2; 
                $event->sheet->getDelegate()->setCellValue('A' . $emptyRow, '');
                $event->sheet->getDelegate()->setCellValue('A' . $totalRow, 'Total');
                $event->sheet->getDelegate()->setCellValue('D' . $totalRow, $this->totalWorkingEarn);
                $event->sheet->getDelegate()->setCellValue('E' . $totalRow, $this->totalLeadEarn);
            },
        ];
    }
}